#----------------------------------------------------------------------
#  GFDM method test - 2d beam under flexural load
#  Author: Andrea Pavan
#  Date: 30/01/2023
#  License: GPLv3-or-later
#----------------------------------------------------------------------
using ElasticArrays;
using LinearAlgebra;
using SparseArrays;
using PyPlot;
include("utils.jl");


#problem definition
l1 = 20.0;      #domain x size
l2 = 1.0;       #domain y size
E = 71.7e9;     #Young modulus
ν = 0.33;       #Poisson ratio
qy = 1e5;       #distributed load on the top

meshSize = 0.2;
#meshSize = 0.05;
surfaceMeshSize = meshSize;
minNeighbors = 10;
minSearchRadius = meshSize;

#μ = 0.5*E/(1+ν);       #Lamè coefficients
#λ = E*ν/((1+ν)*(1-2*ν));


#pointcloud generation
time1 = time();
pointcloud = ElasticArray{Float64}(undef,2,0);      #2xN matrix containing the coordinates [X;Y] of each node
boundaryNodes = Vector{Int};        #indices of the boundary nodes
normals = ElasticArray{Float64}(undef,2,0);     #2xN matrix containing the components [nx;ny] of the normal of each boundary node
for i=0+surfaceMeshSize:surfaceMeshSize:l2-surfaceMeshSize
    append!(pointcloud, [0,i]);
    append!(normals, [-1,0]);
    append!(pointcloud, [l1,i]);
    append!(normals, [1,0]);
end
for i=0+surfaceMeshSize:surfaceMeshSize:l1-surfaceMeshSize
    append!(pointcloud, [i,0]);
    append!(normals, [0,-1]);
    append!(pointcloud, [i,l2]);
    append!(normals, [0,1]);
end
boundaryNodes = collect(range(1,size(pointcloud,2)));
for x=0+meshSize:meshSize:l1-meshSize
    for y=0+meshSize:meshSize:l2-meshSize
        newP = [x,y]+(rand(Float64,2).-0.5).*meshSize/5;
        insertP = true;
        for j in boundaryNodes
            if (newP[1]-pointcloud[1,j])^2+(newP[2]-pointcloud[2,j])^2<(0.75*meshSize)^2
                insertP = false;
            end
        end
        if insertP
            append!(pointcloud, newP);
            append!(normals, [0,0]);
        end
    end
end
internalNodes = collect(range(1+length(boundaryNodes),size(pointcloud,2)));
println("Generated pointcloud in ", round(time()-time1,digits=2), " s");
println("Pointcloud properties:");
println("  Boundary nodes: ",length(boundaryNodes));
println("  Internal nodes: ",length(internalNodes));
println("  Memory: ",memoryUsage(pointcloud,boundaryNodes));


#boundary conditions
N = size(pointcloud,2);     #number of nodes
g1u = zeros(Float64,N);
g2u = zeros(Float64,N);
g3u = zeros(Float64,N);
g1v = zeros(Float64,N);
g2v = zeros(Float64,N);
g3v = zeros(Float64,N);
for i in boundaryNodes
    if pointcloud[1,i]<=0+1e-6
        #left side - u=0, v=0
        g1u[i] = 1.0;
        g2u[i] = 0.0;
        g3u[i] = 0.0;
        g1v[i] = 1.0;
        g2v[i] = 0.0;
        g3v[i] = 0.0;
    #=elseif pointcloud[1,i]>=l1-1e-6
        #right side - u=0, v=0
        g1u[i] = 1.0;
        g2u[i] = 0.0;
        g3u[i] = 0.0;
        g1v[i] = 1.0;
        g2v[i] = 0.0;
        g3v[i] = 0.0;=#
    elseif pointcloud[2,i]>=l2-1e-6
        #top side - qy=-1e5
        g1u[i] = 0.0;
        g2u[i] = 1.0;
        g3u[i] = 0.0;
        g1v[i] = 0.0;
        g2v[i] = 1.0;
        g3v[i] = qy;
    else
        #everywhere else
        g1u[i] = 0.0;
        g2u[i] = 1.0;
        g3u[i] = 0.0;
        g1v[i] = 0.0;
        g2v[i] = 1.0;
        g3v[i] = 0.0;
    end
end
Fx = zeros(Float64,N);      #volumetric loads
Fy = zeros(Float64,N);

#pointcloud plot
#=figure();
plot(pointcloud[1,boundaryNodes],pointcloud[2,boundaryNodes],"r.");
plot(pointcloud[1,internalNodes],pointcloud[2,internalNodes],"k.");
title("Pointcloud plot");
axis("equal");
display(gcf());=#

#boundary conditions plot
#=figure();
scatter(pointcloud[1,:],pointcloud[2,:],c=g1u);
colorbar();
title("Boundary conditions");
axis("equal");
display(gcf());=#


#neighbor search
time2 = time();
neighbors = Vector{Vector{Int}}(undef,N);       #vector containing N vectors of the indices of each node neighbors
Nneighbors = zeros(Int,N);      #number of neighbors of each node
for i=1:N
    searchradius = minSearchRadius;
    while Nneighbors[i]<minNeighbors
        neighbors[i] = Int[];
        #check every other node
        for j=1:N
            if j!=i && all(abs.(pointcloud[:,j]-pointcloud[:,i]).<searchradius)
                push!(neighbors[i],j);
            end
        end
        unique!(neighbors[i]);
        Nneighbors[i] = length(neighbors[i]);
        searchradius += minSearchRadius/2;
    end
end
println("Found neighbors in ", round(time()-time2,digits=2), " s");
println("Connectivity properties:");
println("  Max neighbors: ",maximum(Nneighbors)," (at index ",findfirst(isequal(maximum(Nneighbors)),Nneighbors),")");
println("  Avg neighbors: ",round(sum(Nneighbors)/length(Nneighbors),digits=2));
println("  Min neighbors: ",minimum(Nneighbors)," (at index ",findfirst(isequal(minimum(Nneighbors)),Nneighbors),")");


#neighbors distances and weights
time3 = time();
P = Vector{Array{Float64}}(undef,N);        #relative positions of the neighbors
r2 = Vector{Vector{Float64}}(undef,N);      #relative distances of the neighbors
w = Vector{Vector{Float64}}(undef,N);       #neighbors weights
for i=1:N
    P[i] = Array{Float64}(undef,2,Nneighbors[i]);
    r2[i] = Vector{Float64}(undef,Nneighbors[i]);
    w[i] = Vector{Float64}(undef,Nneighbors[i]);
    for j=1:Nneighbors[i]
        P[i][:,j] = pointcloud[:,neighbors[i][j]]-pointcloud[:,i];
        r2[i][j] = P[i][:,j]'P[i][:,j];
    end
    r2max = maximum(r2[i]);
    for j=1:Nneighbors[i]
        w[i][j] = exp(-6*r2[i][j]/r2max);
        #w[i][j] = 1.0;
    end
end
wpde = 2.0;       #least squares weight for the pde
wbc = 2.0;        #least squares weight for the boundary condition


#least square matrix inversion
Cm = Vector{Matrix}(undef,N);       #stencil coefficients matrices
condV = zeros(N);
condC = zeros(N);
for i in internalNodes
    xj = P[i][1,:];
    yj = P[i][2,:];
    V = zeros(Float64,2+2*Nneighbors[i],12);

    #plane strain
    #=
    for j=1:Nneighbors[i]
        V[j,:] = [1, xj[j], yj[j], xj[j]^2, yj[j]^2, xj[j]*yj[j], 0, 0, 0, 0, 0, 0];
        V[j+Nneighbors[i],:] = [0, 0, 0, 0, 0, 0, 1, xj[j], yj[j], xj[j]^2, yj[j]^2, xj[j]*yj[j]];
    end
    V[1+2*Nneighbors[i],:] = [0, 0, 0, 2*(2+λ/μ), 2, 0, 0, 0, 0, 0, 0, 1+λ/μ];
    V[2+2*Nneighbors[i],:] = [0, 0, 0, 0, 0, 1+λ/μ, 0, 0, 0, 2, 2*(2+λ/μ), 0];
    =#

    #plane stress
    for j=1:Nneighbors[i]
        V[j,:] = [1, xj[j], yj[j], xj[j]^2, yj[j]^2, xj[j]*yj[j],  0, 0, 0, 0, 0, 0];
        V[j+Nneighbors[i],:] = [0, 0, 0, 0, 0, 0,  1, xj[j], yj[j], xj[j]^2, yj[j]^2, xj[j]*yj[j]];
    end
    V[1+2*Nneighbors[i],:] = [0, 0, 0, 2*1/(1-ν^2), 2*0.5*1/(1+ν), 0,  0, 0, 0, 0, 0, 1*ν/(1-ν^2)+0.5*1/(1+ν)];
    V[2+2*Nneighbors[i],:] = [0, 0, 0, 0, 0, 1*ν/(1-ν^2)+0.5*1/(1+ν),  0, 0, 0, 2*0.5*1/(1+ν), 2*1/(1-ν^2), 0];

    condV[i] = cond(V);
    W = Diagonal(vcat(w[i],w[i],wpde,wpde));
    #(Q,R) = qr(W*V);
    #Cm[i] = inv(R)*transpose(Matrix(Q))*W;
    VF = svd(W*V);
    Cm[i] = transpose(VF.Vt)*inv(Diagonal(VF.S))*transpose(VF.U)*W;
    condC[i] = cond(Cm[i]);
end
for i in boundaryNodes
    xj = P[i][1,:];
    yj = P[i][2,:];
    nx = normals[1,i];
    ny = normals[2,i];
    V = zeros(Float64,4+2*Nneighbors[i],12);
    
    #plane strain
    #=
    for j=1:Nneighbors[i]
        V[j,:] = [1, xj[j], yj[j], xj[j]^2, yj[j]^2, xj[j]*yj[j], 0, 0, 0, 0, 0, 0];
        V[j+Nneighbors[i],:] = [0, 0, 0, 0, 0, 0, 1, xj[j], yj[j], xj[j]^2, yj[j]^2, xj[j]*yj[j]];
    end
    V[1+2*Nneighbors[i],:] = [0, 0, 0, 2*(2+λ/μ), 2, 0, 0, 0, 0, 0, 0, 1+λ/μ];
    V[2+2*Nneighbors[i],:] = [0, 0, 0, 0, 0, 1+λ/μ, 0, 0, 0, 2, 2*(2+λ/μ), 0];
    V[3+2*Nneighbors[i],:] = [g1u[i], g2u[i]*normals[1,i]*(2+λ/μ), g2u[i]*normals[2,i], 0, 0, 0, 0, g2u[i]*normals[2,i], g2u[i]*normals[1,i]*λ/μ, 0, 0, 0];
    V[4+2*Nneighbors[i],:] = [0, g2v[i]*normals[2,i]*λ/μ, g2v[i]*normals[1,i], 0, 0, 0, g1v[i], g2v[i]*normals[1,i], g2v[i]*normals[2,i]*(2+λ/μ), 0, 0, 0];
    =#

    #plane stress
    for j=1:Nneighbors[i]
        V[j,:] = [1, xj[j], yj[j], xj[j]^2, yj[j]^2, xj[j]*yj[j],  0, 0, 0, 0, 0, 0];
        V[j+Nneighbors[i],:] = [0, 0, 0, 0, 0, 0,  1, xj[j], yj[j], xj[j]^2, yj[j]^2, xj[j]*yj[j]];
    end
    V[1+2*Nneighbors[i],:] = [0, 0, 0, 2*1/(1-ν^2), 2*0.5*1/(1+ν), 0,  0, 0, 0, 0, 0, 1*ν/(1-ν^2)+0.5*1/(1+ν)];
    V[2+2*Nneighbors[i],:] = [0, 0, 0, 0, 0, 1*ν/(1-ν^2)+0.5*1/(1+ν),  0, 0, 0, 2*0.5*1/(1+ν), 2*1/(1-ν^2), 0];
    V[3+2*Nneighbors[i],:] = [g1u[i], g2u[i]*nx*1/(1-ν^2), g2u[i]*ny*0.5*1/(1+ν), 0, 0, 0,  0, g2u[i]*ny*0.5*1/(1+ν), g2u[i]*nx*1*ν/(1-ν^2), 0, 0, 0];
    V[4+2*Nneighbors[i],:] = [0, g2v[i]*ny*1*ν/(1-ν^2), g2v[i]*nx*0.5*1/(1+ν), 0, 0, 0,  g1v[i], g2v[i]*nx*0.5*1/(1+ν), g2v[i]*ny*1/(1-ν^2), 0, 0, 0];

    condV[i] = cond(V);
    W = Diagonal(vcat(w[i],w[i],wpde,wpde,wbc,wbc));
    #(Q,R) = qr(W*V);
    #Cm[i] = inv(R)*transpose(Matrix(Q))*W;
    VF = svd(W*V);
    Cm[i] = transpose(VF.Vt)*inv(Diagonal(VF.S))*transpose(VF.U)*W;
    condC[i] = cond(Cm[i]);
end
println("Inverted least-squares matrices in ", round(time()-time3,digits=2), " s");
println("Matrix properties:");
println("  Max cond(V): ",maximum(condV));
println("  Avg cond(V): ",sum(condV)/N);
println("  Min cond(V): ",minimum(condV));
println("  Max cond(C): ",maximum(condC));
println("  Avg cond(C): ",sum(condC)/N);
println("  Min cond(C): ",minimum(condC));

#condition number distribution
#=figure();
plt.hist(condC,10);
title("Condition number distribution");
xlabel("cond(C)");
ylabel("Absolute frequency");
display(gcf());=#


#matrix assembly
time4 = time();
rows = Int[];
cols = Int[];
vals = Float64[];
for i=1:N
    #u equation
    push!(rows, i);
    push!(cols, i);
    push!(vals, 1);
    for j=1:Nneighbors[i]
        push!(rows, i);
        push!(cols, neighbors[i][j]);
        push!(vals, -Cm[i][1,j]);
        push!(rows, i);
        push!(cols, N+neighbors[i][j]);
        push!(vals, -Cm[i][1,j+Nneighbors[i]]);
    end
    #v equation
    push!(rows, N+i);
    push!(cols, N+i);
    push!(vals, 1);
    for j=1:Nneighbors[i]
        push!(rows, N+i);
        push!(cols, neighbors[i][j]);
        push!(vals, -Cm[i][7,j]);
        push!(rows, N+i);
        push!(cols, N+neighbors[i][j]);
        push!(vals, -Cm[i][7,j+Nneighbors[i]]);
    end
end
Mm = sparse(rows,cols,vals,2*N,2*N);
println("Completed matrix assembly in ", round(time()-time4,digits=2), " s");


#linear system solution
time5 = time();
bm = zeros(2*N);       #rhs vector
for i=1:N
    bm[i] = 0;
    bm[N+i] = 0;
    #bm[i] += -Cm[i][1,1+2*Nneighbors[i]]*Fx[i]/μ -Cm[i][1,2+2*Nneighbors[i]]*Fy[i]/μ;       #volumetric loads (plane strain)
    #bm[N+i] += -Cm[i][7,1+2*Nneighbors[i]]*Fx[i]/μ -Cm[i][7,2+2*Nneighbors[i]]*Fy[i]/μ;
    bm[i] += -Cm[i][1,1+2*Nneighbors[i]]*Fx[i]/E -Cm[i][1,2+2*Nneighbors[i]]*Fy[i]/E;       #volumetric loads (plane stress)
    bm[N+i] += -Cm[i][7,1+2*Nneighbors[i]]*Fx[i]/E -Cm[i][7,2+2*Nneighbors[i]]*Fy[i]/E;
end
for i in boundaryNodes
    #bm[i] += Cm[i][1,end-1]*g3u[i]/μ + Cm[i][1,end]*g3v[i]/μ;      #plane strain
    #bm[N+i] += Cm[i][7,end-1]*g3u[i]/μ + Cm[i][7,end]*g3v[i]/μ;
    bm[i] += Cm[i][1,end-1]*g3u[i]/E + Cm[i][1,end]*g3v[i]/E;      #plane stress
    bm[N+i] += Cm[i][7,end-1]*g3u[i]/E + Cm[i][7,end]*g3v[i]/E;
end
sol = Mm\bm;
#using IterativeSolvers;
#using Preconditioners;
#PM = DiagonalPreconditioner(Mm);
#sol = bicgstabl(Mm,bm,Pl=PM,verbose=true);
println("Linear system solved in ", round(time()-time5,digits=2), " s");


#displacement plot
u = sol[1:N];
v = sol[N+1:end];
figure();
scatter(pointcloud[1,:],pointcloud[2,:],c=u,cmap="jet");
colorbar();
title("Numerical solution - x displacement");
axis("equal");
display(gcf());
figure();
scatter(pointcloud[1,:],pointcloud[2,:],c=v,cmap="jet");
colorbar();
title("Numerical solution - y displacement");
axis("equal");
display(gcf());


#deformed beam plot
figure();
#scatter(pointcloud[1,:]+u*200,pointcloud[2,:]+v*200,c=sqrt.(u.^2+v.^2),cmap="Oranges");
scatter(pointcloud[1,:]+3*u,pointcloud[2,:]+3*v,c=v,cmap="Oranges");
colorbar();
title("Numerical solution - deflection");
axis("equal");
display(gcf());

println("\nEuler-Bernoulli beam theory:");
println("> vmax = ",qy*(l1^4)/(8*E*(l2^4)/12));
println("GFDM:");
println("> umax = ",maximum(abs.(u)));
println("> vmax = ",maximum(abs.(v)));
